<?php
/* --------------------------------------------------------------
	PayPalInstallments.inc.php 2018-10-17
	Gambio GmbH
	http://www.gambio.de
	Copyright (c) 2017 Gambio GmbH
	Released under the GNU General Public License (Version 2)
	[http://www.gnu.org/licenses/gpl-2.0.html]
	--------------------------------------------------------------
*/

class PayPalInstallments
{
	protected $logger;
	
	public function __construct()
	{
		$this->setLogger(MainFactory::create('PayPalLogger'));
	}
	
	
	/**
	 * Sets logger instance.
	 *
	 * Note that this is NOT a PSR-3 compatible logging mechanism.
	 *
	 * @param \PayPalLogger $logger
	 */
	public function setLogger(PayPalLogger $logger)
	{
		$this->logger = $logger;
	}
	
	/**
	 * Retrieves data for Upstream Presentment.
	 *
	 * @param int    $amount
	 * @param string $currency
	 * @param        $country
	 *
	 * @return \stdClass
	 * @throws \Exception
	 */
	public function getInstallmentInfo($amount = 0, $currency = 'EUR', $country = STORE_COUNTRY)
	{
	    $amountString = number_format((float)$amount, 2, '.', '');
		$financingOptions = json_encode([
				'financing_country_code' => $country,
				'transaction_amount' => [
					'value'         => $amountString,
					'currency_code' => $currency,
				],
			]);
		$this->logger->iup_notice(sprintf('fetching data for upstream presentment, %s %s (%s)', $amountString, $currency, $country));
		$request      = MainFactory::create('PayPalRestRequest', 'POST', '/v1/credit/calculated-financing-options', $financingOptions, 'inst');
		$service      = MainFactory::create('PayPalRestService');
		$service->setLogRawResponseEnabled(false);
		$service->setLogDecodedResponseEnabled(false);
		$response     = $service->performRequest($request);
		$responseBody = $response->getResponseObject();
		if($responseBody !== false)
		{
			$this->logResponse($responseBody);
			usort($responseBody->financing_options[0]->qualifying_financing_options, function($a, $b) {
				return ($a->credit_financing->term < $b->credit_financing->term) ? -1 : 1;
			});
		}
		else
		{
			$this->logger->iup_notice('ERROR retrieving data for upstream presentment!');
		}
		return $responseBody;
	}
	
	
	/**
	 * Creates a log entry from a financing options response
	 *
	 * @param \stdClass $logResponse
	 */
	protected function logResponse(stdClass $logResponse)
	{
		if(isset($logResponse->financing_options) && is_array($logResponse->financing_options))
		{
			$financingOptions = $logResponse->financing_options;
			if(isset($financingOptions[0]->qualifying_financing_options) && is_array($financingOptions[0]->qualifying_financing_options))
			{
				$qualifyingOptions = $financingOptions[0]->qualifying_financing_options;
				$logMessage = "Financing options:\n";
				foreach($qualifyingOptions as $qOption)
				{
					$logMessage .= sprintf("%s %s: %s%% %s %s, %s total\n",
					                       $qOption->credit_financing->product,
					                       $qOption->credit_financing->financing_code,
					                       $qOption->credit_financing->apr,
					                       $qOption->credit_financing->term,
					                       $qOption->credit_financing->interval,
					                       $qOption->total_cost->value . ' ' . $qOption->total_cost->currency_code
						);
				}
				$this->logger->iup_notice($logMessage);
			}
			else
			{
				$this->logger->iup_notice('ERROR: PayPal did not return any qualifying financing options!');
			}
		}
		else
		{
			$this->logger->iup_notice('ERROR: PayPal did not return any financing options!');
		}
	}

	/** takes an array of financing options and returns the one with the highest APR and lowest monthly payment */
	public function getRepresentativeOption($financingOptions)
	{
		$representativeOption = false;
		$highestAPR = 0;
		$lowestMonthly = 999999;
		foreach($financingOptions as $financingOption)
		{
			if($financingOption->credit_financing->apr >= $highestAPR)
			{
				$highestAPR = $financingOption->credit_financing->apr;
				if($financingOption->monthly_payment->value <= $lowestMonthly)
				{
					$representativeOption = $financingOption;
					$lowestMonthly = $financingOption->monthly_payment->value;
				}
			}
		}
		return $representativeOption;
	}

	public function getFinancingByCode($financingCode)
	{
		$request      = MainFactory::create('PayPalRestRequest', 'GET', '/v1/credit/credit-financing/' . $financingCode, 'inst');
		$service      = MainFactory::create('PayPalRestService');
		$response     = $service->performRequest($request);
		$responseBody = $response->getResponseObject();
		return $responseBody;
	}

	/**
	 * return an array with keys 'amount' and 'currency' representing the minimum amount for which installments are available
	 *
	 * @return array keys: 'amount' (double), 'currency' (3-letter currency symbol)
	 */
	public function getMinimumAmount()
	{
		return ['amount' => 99.00, 'currency' => 'EUR'];
	}

	/**
	 * return an array with keys 'amount' and 'currency' representing the maximum amount for which installments are available
	 *
	 * @return array keys: 'amount' (double), 'currency' (3-letter currency symbol)
	 */
	public function getMaximumAmount()
	{
		return ['amount' => 5000.00, 'currency' => 'EUR'];
	}
}
